<?php
/* --------------------------------------------------------------
 AbstractAction.php 2020-02-27
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\Http;

use Gambio\Core\Application\ValueObjects\Path;
use Gambio\Core\Application\ValueObjects\Url;
use Slim\Http\Response;
use Slim\Http\ServerRequest;

/**
 * Class AbstractAction
 * @package Gambio\Admin\Application\Http
 */
abstract class AbstractAction
{
    /**
     * @var ServerRequest
     */
    protected $request;
    
    /**
     * @var Response
     */
    protected $response;
    
    /**
     * @var Url
     */
    protected $url;
    
    /**
     * @var Path
     */
    protected $path;
    
    
    /**
     * Action initialization.
     *
     * The DI container automatically injects the given dependencies for all actions.
     *
     * @param Url  $url
     * @param Path $path
     */
    public function initialize(Url $url, Path $path): void
    {
        $this->url  = $url;
        $this->path = $path;
    }
    
    
    /**
     * Executing the action.
     *
     * This method is called from the slim framework. Clients just have to implement the process
     * method to execute their actions logic. Dependencies, like request, response and url/path informations
     * are automatically injected.
     *
     * @param ServerRequest $request
     * @param Response      $response
     *
     * @return Response
     */
    public function __invoke(ServerRequest $request, Response $response): Response
    {
        $this->request  = $request;
        $this->response = $response;
        
        return $this->process();
    }
    
    
    /**
     * Processes the action.
     *
     * @return Response
     */
    abstract protected function process(): Response;
}